<?php
/**
 *
 * SugarCRM Community Edition is a customer relationship management program developed by
 * SugarCRM, Inc. Copyright (C) 2004-2013 SugarCRM Inc.
 *
 * SuiteCRM is an extension to SugarCRM Community Edition developed by SalesAgility Ltd.
 * Copyright (C) 2011 - 2021 SalesAgility Ltd.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Affero General Public License version 3 as published by the
 * Free Software Foundation with the addition of the following permission added
 * to Section 15 as permitted in Section 7(a): FOR ANY PART OF THE COVERED WORK
 * IN WHICH THE COPYRIGHT IS OWNED BY SUGARCRM, SUGARCRM DISCLAIMS THE WARRANTY
 * OF NON INFRINGEMENT OF THIRD PARTY RIGHTS.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Affero General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Affero General Public License along with
 * this program; if not, see http://www.gnu.org/licenses or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 *
 * You can contact SugarCRM, Inc. headquarters at 10050 North Wolfe Road,
 * SW2-130, Cupertino, CA 95014, USA. or at email address contact@sugarcrm.com.
 *
 * The interactive user interfaces in modified source and object code versions
 * of this program must display Appropriate Legal Notices, as required under
 * Section 5 of the GNU Affero General Public License version 3.
 *
 * In accordance with Section 7(b) of the GNU Affero General Public License version 3,
 * these Appropriate Legal Notices must retain the display of the "Powered by
 * SugarCRM" logo and "Supercharged by SuiteCRM" logo. If the display of the logos is not
 * reasonably feasible for technical reasons, the Appropriate Legal Notices must
 * display the words "Powered by SugarCRM" and "Supercharged by SuiteCRM".
 */

namespace SuiteCRM\Utility\AntiMalware\Providers;

use Psr\Log\LoggerAwareInterface;
use Psr\Log\LoggerInterface;
use SuiteCRM\Exception\Exception;
use SuiteCRM\Exception\InvalidArgumentException;
use SuiteCRM\Exception\MalwareFoundException;
use SuiteCRM\Interfaces\AntiMalwareFileScanner;
use SuiteCRM\Utility\AntiMalware\AntiMalwareTrait;
use SuiteCRM\Utility\Configuration;
use SuiteCRM\Utility\SuiteLogger;

/**
 * Class ClamTCP
 * @package SuiteCRM\Utility\AntiMalware\Providers
 */
#[\AllowDynamicProperties]
class ClamTCP implements AntiMalwareFileScanner, LoggerAwareInterface
{
    use AntiMalwareTrait;

    /**
     * @var bool $enabled
     */
    private $enabled = false;

    /**
     * @var string $ip
     */
    private $ip = '127.0.0.1';

    /**
     * @var int $port
     */
    private $port = 3306;

    /**
     * @var string $type local, remote
     */
    private $type = 'local';

    /**
     * @see clamconf: StreamMaxLength
     * @var int $chunkSize
     */
    private $chunkSize = 26214400;

    /**
     * @see clamconf: MaxFileSize
     * @var int $clamAvMax
     */
    private $clamAvMax = 26214400;

    /**
     * @var  LoggerInterface
     */
    private $logger;

    /**
     * constructor.
     */
    public function __construct()
    {
        $globalConfig = new Configuration();
        $configuration = $globalConfig['anti_malware_scanners'][self::class] ?? null;

        if (!empty($configuration)) {
            $this->enabled = $configuration['enabled'] ?? $this->enabled;
            $this->ip = $configuration['options']['ip'] ?? $this->ip;
            $this->port = $configuration['options']['port'] ?? $this->port;
            $this->type = $configuration['options']['type'] ?? $this->type;
        }

        $this->logger = new SuiteLogger();
    }

    /**
     * Is the Anti-Malware FileScanner enabled
     *
     * @return bool true === yes, false === no
     */
    public function isAntiMalwareScannerAvailable(): bool
    {
        return $this->isEnabled();
    }

    /**
     * Scan path for malware
     *
     * @param string $path
     * @throws InvalidArgumentException
     * @throws Exception
     */
    public function scanFilePath(string $path): void
    {
        $this->validatePath($path);

        switch ($this->type) {
            case 'local':
                $this->scanLocalFile($path);
                break;
            case 'remote':
                $this->streamScan($path);
                break;
            default:
                throw new InvalidArgumentException(
                    '[ClamAntiVirus][scanPathForMalware] $type should be a recognised value'
                );
        }
    }

    /**
     * Requests clamd scan of local file
     *
     * @param string $path
     * @throws Exception
     */
    protected function scanLocalFile(string $path): void
    {
        $response = $this->sendCommand('SCAN ' . $path);
        $this->processResponse($path, $response);
    }

    /**
     * Sends TCP Stream of local file to remote clamd for scanning
     * @param $path
     * @throws Exception
     * @noinspection PhpComposerExtensionStubsInspection
     */
    protected function streamScan($path): void
    {
        if ($scan_fh = fopen($path, 'rb')) {

            $socket = $this->getSocket();

            $chunkSize = filesize($path) < $this->chunkSize ? filesize($path) : $this->chunkSize;

            // do not scan empty file
            if ($chunkSize === 0) {
                return;
            }

            $command = "zINSTREAM\0";
            socket_send($socket, $command, strlen($command), 0);

            while (!feof($scan_fh)) {
                $data = fread($scan_fh, $chunkSize);
                $packet = pack(sprintf("Na%d", strlen($data)), strlen($data), $data);
                socket_send($socket, $packet, strlen($packet), 0);
            }

            $packet = pack("Nx", 0);
            socket_send($socket, $packet, strlen($packet), 0);
            socket_recv($socket, $response, $this->clamAvMax, 0);
            socket_close($socket);

            $this->processResponse($path, $response);
        } else {
            throw new Exception(
                '[ClamAntiVirus][scanPathForMalware] cannot open file for scanning'
            );
        }
    }

    /**
     * @param $command
     * @return string|null
     * @throws Exception
     * @noinspection PhpComposerExtensionStubsInspection
     */
    protected function sendCommand($command): ?string
    {
        $return = null;
        $socket = $this->getSocket();

        socket_send($socket, $command, strlen((string) $command), 0);
        socket_recv($socket, $return, $this->clamAvMax, 0);
        socket_close($socket);

        return $return;
    }

    /**
     * @param string $path - location in the file system
     * @param $response - clamd response
     * @throws MalwareFoundException
     */
    protected function processResponse(string $path, $response): void
    {
        $response = explode(':', $response);
        $result = trim(end($response));

        if ($result !== "OK") {
            $this->logger->emergency($result);
            throw new MalwareFoundException(
                '[ClamAntiVirus][MalwareDetected]' . $path . $result
            );
        }
    }

    /**
     * @throws Exception
     * @noinspection PhpComposerExtensionStubsInspection
     */
    protected function getSocket()
    {
        $socket = @socket_create(AF_INET, SOCK_STREAM, 0);
        $status = @socket_connect($socket, $this->ip, $this->port);

        if (!$status) {
            throw new Exception(
                '[ClamAntiVirus][scanPathForMalware] cannot connect to clamAV daemon'
            );
        }

        return $socket;
    }

    /**
     * @throws InvalidArgumentException
     */
    private function validatePath($path): void
    {
        if (!is_string($path)) {
            throw new InvalidArgumentException(
                '[ClamAntiVirus][scanPathForMalware] $path should be a string'
            );

        }

        if (!file_exists($path)) {
            throw new InvalidArgumentException(
                '[ClamAntiVirus][scanPathForMalware] ' . $path . ' not found'
            );
        }
    }

    /**
     * @param LoggerInterface $logger
     */
    public function setLogger(LoggerInterface $logger)
    {
        $this->logger = $logger;
    }

    /**
     * @return boolean
     */
    private function isEnabled(): bool
    {
        return $this->enabled;
    }

}
